import argparse
import builtins
import math
import os
import random
import shutil
import time
import warnings
import timm
from functools import partial
os.environ["HDF5_USE_FILE_LOCKING"] = "FALSE"
from moco.builder import *
from moco.loader import *
from moco.utils import *
import moco.builder
import moco.loader
import moco.utils

import torch
import torch.backends.cudnn as cudnn
import torch.distributed as dist
import torch.multiprocessing as mp
import torch.nn as nn
import torch.nn.parallel
import torch.optim
import torch.utils.data
import torch.utils.data.distributed
from torch.nn.utils import clip_grad_norm_
from torch.amp import autocast, GradScaler

from braincog.model_zoo.sew_resnet import *
from braincog.model_zoo.vgg_snn import *
import wandb

parser = argparse.ArgumentParser(description="Unsupervised SNN Pre-Training")

# 参数 ================================================================================

parser.add_argument(
    "--model_config",
    default=None,
    help="model config file",
)
parser.add_argument(
    "--dataset_config",
    default=None,
    help="dataset config file",
)
parser.add_argument(
    "--method_config",
    default=None,
    help="method config file",
)
parser.add_argument(
    "-a",
    "--arch",
    metavar="ARCH",
    default="resnet50",
    choices=timm.list_models(),
    help="model architecture: " + " | ".join(timm.list_models()) + " (default: resnet50)",
)
parser.add_argument(
    "--dataset", type=str,  default="cifar10", help="which dataset is used"
)
parser.add_argument(
    "--dataset_path", type=str,  default=None, help="Where is the dataset"
)
parser.add_argument(
    "--cache_path", type=str,  default=None, help="Where is the dataset cache"
)
parser.add_argument(
    "--method", type=str,  default="MoCo_TET", help="which method is used"
)
parser.add_argument(
    "-j",
    "--workers",
    default=16,
    type=int,
    metavar="N",
    help="number of data loading workers (default: 16)",
)
parser.add_argument(
    "-v",
    "--val-workers",
    default=16,
    type=int,
    metavar="N",
    help="number of val data loading workers (default: 16)",
)
parser.add_argument(
    "--epochs", default=200, type=int, metavar="N", help="number of total epochs to run"
)
parser.add_argument(
    "--start-epoch",
    default=0,
    type=int,
    metavar="N",
    help="manual epoch number (useful on restarts)",
)
parser.add_argument(
    "-b",
    "--batch-size",
    default=256,
    type=int,
    metavar="N",
    help="mini-batch size (default: 256), this is the total "
    "batch size of all GPUs on the current node when "
    "using Data Parallel or Distributed Data Parallel",
)
parser.add_argument(
    "--size",
    default=128,
    type=int,
    help="image size (default: 128)",
)
parser.add_argument(
    "--lr",
    "--learning-rate",
    default=0.03,
    type=float,
    metavar="LR",
    help="initial learning rate",
    dest="lr",
)
parser.add_argument(
    "--optimizer",
    default="AdamW",
    type=str,
    help="optimizer",
)
parser.add_argument(
    "--schedule",
    default=[60, 80],
    nargs="*",
    type=int,
    help="learning rate schedule (when to drop lr by 10x)",
)
parser.add_argument(
    "--momentum", default=0.9, type=float, metavar="M", help="momentum of SGD solver"
)
parser.add_argument(
    "--wd",
    "--weight-decay",
    default=1e-4,
    type=float,
    metavar="W",
    help="weight decay (default: 1e-4)",
    dest="weight_decay",
)
parser.add_argument(
    "--save-path",
    default='.',
    type=str,
    help="weight save path",
)
parser.add_argument(
    "-p",
    "--print-freq",
    default=10,
    type=int,
    metavar="N",
    help="print frequency (default: 10)",
)
parser.add_argument(
    "--save-freq",
    default=999,
    type=int,
    metavar="N",
    help="save frequency (default: 10)",
)
parser.add_argument(
    "--resume",
    default="",
    type=str,
    metavar="PATH",
    help="path to latest checkpoint (default: none)",
)
parser.add_argument(
    "--pretrained",
    default="",
    type=str,
    help="path to pretrained checkpoint to load before training",
)
parser.add_argument(
    "--world-size",
    default=-1,
    type=int,
    help="number of nodes for distributed training",
)
parser.add_argument(
    "--rank", default=-1, type=int, help="node rank for distributed training"
)
parser.add_argument(
    "--dist-url",
    default="tcp://224.66.41.62:23456",
    type=str,
    help="url used to set up distributed training",
)
parser.add_argument(
    "--dist-backend", default="nccl", type=str, help="distributed backend"
)
parser.add_argument(
    "--seed", default=None, type=int, help="seed for initializing training. "
)
parser.add_argument(
    "--gpu", default=None, type=int, help="GPU id to use."
)
parser.add_argument(
    "--multiprocessing-distributed",
    action="store_true",
    help="Use multi-processing distributed training to launch "
    "N processes per node, which has N GPUs. This is the "
    "fastest way to use PyTorch for either single node or "
    "multi node data parallel training",
)
parser.add_argument(
    "--suffix", type=str,  default="", help="which method is used"
)

# used for SNN
parser.add_argument(
    "--step", type=int,  default=4, help="which method is used"
)
parser.add_argument(
    "--layer_by_layer", action="store_true", help="Is layer by layer used"
)

#used for pretraining
parser.add_argument(
    "--output-dim", default=128, type=int, help="feature dimension (default: 128)"
)

#usd for moco
parser.add_argument(
    "--moco-k",
    default=65536,
    type=int,
    help="queue size; number of negative keys (default: 65536)",
)
parser.add_argument(
    "--moco-m",
    default=0.999,
    type=float,
    help="moco momentum of updating key encoder (default: 0.999)",
)
parser.add_argument(
    "--moco-t", default=0.07, type=float, help="softmax temperature (default: 0.07)"
)

# options for moco v2
parser.add_argument(
    "--mlp", action="store_true", help="use mlp head"
)
parser.add_argument(
    "--aug-plus", action="store_true", help="use moco v2 data augmentation"
)
parser.add_argument(
    "--cos", action="store_true", help="use cosine lr schedule"
)

# sync batch norm
parser.add_argument(
    "--sync-bn", action="store_true", help="use SyncBatchNorm across GPUs"
)

# used for SimSiam or PredNext
parser.add_argument(
    "--pred-dim", type=int,  default=512, help="pred dimension (default: 512)"
)
parser.add_argument(
    "--pred-step", type=int,  default=1, help="pred step (default: 1)"
)
parser.add_argument(
    "--pred-alpha", type=float,  default=0.5, help="pred alpha (default: 0.5)"
)

parser.add_argument(
    "--frames-per-clip", type=int,  default=10, help="frames per clip (default: 10)"
)
parser.add_argument(
    "--frame-rate", type=float,  default=1, help="frame rate (default: 25/8)"
)
parser.add_argument(
    "--knn-k", type=int,  default=200, help="knn k (default: 200)"
)
parser.add_argument(
    "--knn-t", type=float,  default=0.1, help="knn t (default: 0.1)"
)
parser.add_argument(
    "--knn-freq", type=int,  default=4, help="knn freq (default: 10)"
)
parser.add_argument(
    "--knn-max-samples", type=int,  default=99999999, help="knn max samples (default: 99999999)"
)
parser.add_argument(
    "--warmup-epochs", type=int,  default=0, help="warmup epochs (default: 10)"
)
parser.add_argument(
    "--node-type", type=str,  default="LIFNode", help="node type (default: LIFNode)"
)
parser.add_argument(
    "--threshold", type=float,  default=0.5, help="threshold (default: 0.5)"
)
parser.add_argument(
    "--tau", type=float,  default=2, help="tau (default: 2)"
)
parser.add_argument(
    "--channel-first", action="store_true", help="use channel first"
)
parser.add_argument(
    "--num-clips", type=int,  default=1, help="number of clips (default: 1)"
)
parser.add_argument(
    "--train-num-clips", type=int,  default=1, help="number of clips (default: 1)"
)
parser.add_argument(
    "--fc-lr-scale", type=float,  default=1.0, help="fc lr scale (default: 1.0)"
)
parser.add_argument(
    "--multi-train-sample", action="store_true", help="use multi train sample"
)
# 主函数 ================================================================================

def main():
    args = parser.parse_args()
    args = load_config(args)
        
    if args.seed is not None:
        random.seed(args.seed)
        torch.manual_seed(args.seed)
        cudnn.deterministic = True
        warnings.warn(
            "You have chosen to seed training. "
            "This will turn on the CUDNN deterministic setting, "
            "which can slow down your training considerably! "
            "You may see unexpected behavior when restarting "
            "from checkpoints."
        )

    if args.gpu is not None:
        warnings.warn(
            "You have chosen a specific GPU. This will completely "
            "disable data parallelism."
        )

    if args.dist_url == "env://" and args.world_size == -1:
        args.world_size = int(os.environ["WORLD_SIZE"])

    args.distributed = args.world_size > 1 or args.multiprocessing_distributed

    ngpus_per_node = torch.cuda.device_count()
    args.ngpus_per_node = ngpus_per_node
    if args.multiprocessing_distributed:
        # Since we have ngpus_per_node processes per node, the total world_size
        # needs to be adjusted accordingly
        args.world_size = ngpus_per_node * args.world_size
        # Use torch.multiprocessing.spawn to launch distributed processes: the
        # main_worker process function
        mp.spawn(main_worker, nprocs=ngpus_per_node, args=(ngpus_per_node, args))
    else:
        # Simply call main_worker function
        main_worker(args.gpu, ngpus_per_node, args)

def main_worker(gpu, ngpus_per_node, args):
    args.gpu = gpu

    # suppress printing if not master
    if args.multiprocessing_distributed and args.gpu != 0:

        def print_pass(*args):
            pass

        builtins.print = print_pass

    if args.gpu is not None:
        print("Use GPU: {} for training".format(args.gpu))

    if args.distributed:
        if args.dist_url == "env://" and args.rank == -1:
            args.rank = int(os.environ["RANK"])
        if args.multiprocessing_distributed:
            # For multiprocessing distributed training, rank needs to be the
            # global rank among all the processes
            args.rank = args.rank * ngpus_per_node + gpu
        dist.init_process_group(
            backend=args.dist_backend,
            init_method=args.dist_url,
            world_size=args.world_size,
            rank=args.rank,
        )
    # create model
    print("=> creating model '{}'".format(args.arch)) 
    
    if is_SNN(args.arch):
        model_arch=partial(
            timm.models.model_entrypoint(args.arch),
            dataset=args.dataset,
            layer_by_layer=args.layer_by_layer,
            is_cifar=True if "cifar" in args.dataset else False,
            step=args.step
            )
    else:
        model_arch=partial(
            timm.models.model_entrypoint(args.arch)
            )
        
    model = eval(f"moco.builder.{args.method}")(
        model_arch,
        dim=args.output_dim,
        K=args.moco_k,
        m=args.moco_m,
        T=args.moco_t,
        mlp=args.mlp,
        pred_dim=args.pred_dim,
        args=args,
    )
    
    
    # convert to SyncBatchNorm before wrapping with DDP
    if args.distributed and args.sync_bn:
        print("=> using SyncBatchNorm")
        model = torch.nn.SyncBatchNorm.convert_sync_batchnorm(model)
    print(model)

    # load from pre-trained, before DistributedDataParallel constructor
    if args.pretrained:
        if os.path.isfile(args.pretrained):
            print("=> loading pretrained checkpoint '{}'".format(args.pretrained))
            checkpoint = torch.load(args.pretrained, map_location="cpu", weights_only=False)
            try:
                state_dict = checkpoint["state_dict"]
            except:
                state_dict = checkpoint.get("model", checkpoint)
            # remove leading 'module.' if present (loading into non-DDP model)
            if all(k.startswith("module.") for k in state_dict.keys()):
                state_dict = {k[len("module."):]: v for k, v in state_dict.items()}
            state_dict = {'.'.join(p for p in k.split('.') if p != 'module'): v for k, v in state_dict.items()}  
            state_dict = {'.'.join(p for p in k.split('.') if p != 'encoder_q'): v for k, v in state_dict.items()}   
            state_dict = {k: v for k, v in state_dict.items() if "fc" not in k}  
            msg = model.encoder_q.load_state_dict(state_dict, strict=False)
            try:
                missing = list(msg.missing_keys)
                missing = [k for k in missing if "alpha" not in k and "threshold" not in k]
                unexpected = list(msg.unexpected_keys)
                if missing or unexpected:
                    print(f"=> loaded with missing keys: {missing} \n\n unexpected keys: {unexpected}")
            except Exception:
                pass
            print("=> pretrained weights loaded")
        else:
            print("=> no pretrained checkpoint found at '{}'".format(args.pretrained))


    args.init_lr = args.lr * args.batch_size / 256
    if args.distributed: 
        # For multiprocessing distributed, DistributedDataParallel constructor
        # should always set the single device scope, otherwise,
        # DistributedDataParallel will use all available devices.
        if args.gpu is not None:
            torch.cuda.set_device(args.gpu)
            model.cuda(args.gpu)
            # When using a single GPU per process and per
            # DistributedDataParallel, we need to divide the batch size
            # ourselves based on the total number of GPUs we have
            args.batch_size = int(args.batch_size / ngpus_per_node)
            args.workers = int((args.workers + ngpus_per_node - 1) / ngpus_per_node)
            args.val_workers = int((args.val_workers + ngpus_per_node - 1) / ngpus_per_node)
            model = torch.nn.parallel.DistributedDataParallel(
                model, device_ids=[args.gpu]
            )
        else:
            model.cuda()
            # DistributedDataParallel will divide and allocate batch_size to all
            # available GPUs if device_ids are not set
            model = torch.nn.parallel.DistributedDataParallel(model)
    elif args.gpu is not None:
        torch.cuda.set_device(args.gpu)
        model = model.cuda(args.gpu)
        # comment out the following line for debugging
        raise NotImplementedError("Only DistributedDataParallel is supported.")
    # else:
    #     # AllGather implementation (batch shuffle, queue update, etc.) in
    #     # this code only supports DistributedDataParallel.
    #     raise NotImplementedError("Only DistributedDataParallel is supported.")

    # define loss function (criterion) and optimizer
    criterion = nn.CrossEntropyLoss().cuda(args.gpu)
    args.criterion = criterion
    
    if args.optimizer == "SGD":
        optimizer = torch.optim.SGD(
            model.parameters(), args.lr, momentum=args.momentum, weight_decay=args.weight_decay
        )
    elif args.optimizer == "AdamW":
        optimizer = torch.optim.AdamW(
            model.parameters(), args.lr, weight_decay=args.weight_decay
        )
    else:
        optimizer = eval(f"torch.optim.{args.optimizer}")(
            model.parameters(), args.lr, weight_decay=args.weight_decay
        )

    # optionally resume from a checkpoint
    if args.resume:
        if os.path.isfile(args.resume):
            print("=> loading checkpoint '{}'".format(args.resume))
            if args.gpu is None:
                checkpoint = torch.load(args.resume,weights_only=False)
            else:
                # Map model to be loaded to specified single gpu.
                loc = "cuda:{}".format(args.gpu)
                checkpoint = torch.load(args.resume, map_location=loc,weights_only=False)
            args.start_epoch = checkpoint["epoch"]
            model.load_state_dict(checkpoint["state_dict"])
            optimizer.load_state_dict(checkpoint["optimizer"])
            print(
                "=> loaded checkpoint '{}' (epoch {})".format(
                    args.resume, checkpoint["epoch"]
                )
            )
        else:
            print("=> no checkpoint found at '{}'".format(args.resume))

    cudnn.benchmark = True

 
    train_dataset, knn_train_dataset, test_dataset = eval(f"get_{args.dataset}_datasets")(args)
    args.best_losses=999999
    
    
    if args.distributed:
        train_sampler = torch.utils.data.distributed.DistributedSampler(train_dataset)
        knn_train_sampler = torch.utils.data.distributed.DistributedSampler(knn_train_dataset)
    else:
        train_sampler = None

    train_loader = torch.utils.data.DataLoader(
        train_dataset,
        batch_size=args.batch_size,
        shuffle=(train_sampler is None),
        num_workers=args.workers,
        pin_memory=True,
        sampler=train_sampler,
        drop_last=True,
        persistent_workers=True,
        prefetch_factor=1
    )

    knn_train_loader = torch.utils.data.DataLoader(
        knn_train_dataset,
        batch_size=args.batch_size,
        shuffle=(knn_train_sampler is None),
        num_workers=args.val_workers,
        pin_memory=True,
        sampler=knn_train_sampler,
        drop_last=True,
        persistent_workers=True,
        prefetch_factor=1
    )
    
    test_loader = torch.utils.data.DataLoader(
        test_dataset,
        batch_size=args.batch_size,
        shuffle=False,
        num_workers=args.val_workers,
        pin_memory=True,
        persistent_workers=True,
        prefetch_factor=1
    )
    
    if not args.multiprocessing_distributed or (
            args.multiprocessing_distributed and args.rank % ngpus_per_node == 0
        ):
        wandb.init(
            project="Pretrain_CrossTP",
            config=args,
            
        )
    scaler = GradScaler()            
    for epoch in range(args.start_epoch, args.epochs):
        if args.distributed:
            train_sampler.set_epoch(epoch)
        adjust_learning_rate(optimizer, epoch, args)

        # train for one epoch
        train(train_loader, knn_train_loader, test_loader, model, criterion, optimizer, scaler, epoch, args)

        if not args.multiprocessing_distributed or (
            args.multiprocessing_distributed and args.rank % ngpus_per_node == 0
        ):
            if args.best:
                try:
                    save_filename=os.path.join(args.save_path,f"checkpoints/pretrain/{args.dataset}/{args.method}/{args.arch}/{wandb.run.id}{args.suffix}.pth.tar")
                except:
                    save_filename=os.path.join(args.save_path,f"checkpoints/pretrain/{args.dataset}/{args.method}/{args.arch}/{wandb.get_run().public.run_id}{args.suffix}.pth.tar")
                save_checkpoint(
                    {
                        "epoch": epoch + 1,
                        "args": args,
                        "state_dict": model.state_dict(),
                        "optimizer": optimizer.state_dict(),
                    },
                    filename=save_filename,
                )
    if args.distributed:
        dist.destroy_process_group()

def train(train_loader, knn_train_loader, test_loader, model, criterion, optimizer, scaler, epoch, args):
    batch_time = AverageMeter("Time", ":6.3f")
    data_time = AverageMeter("Data", ":6.3f")
    losses = AverageMeter("Loss", ":.4e")
    losses_full = AverageMeter("Loss_full", ":.4e")
    losses_next = AverageMeter("Loss_next", ":.4e")
    losses_clip = AverageMeter("Loss_clip", ":.4e")
    top1 = AverageMeter("Acc@1", ":6.2f")
    top5 = AverageMeter("Acc@5", ":6.2f")
    progress = ProgressMeter(
        len(train_loader),
        [batch_time, data_time, losses, top1, top5],
        prefix="Epoch: [{}]".format(epoch),
    )

    # switch to train mode
    model.train()

    end = time.time()
    for i, (images, _) in enumerate(train_loader):
        # measure data loading time
        data_time.update(time.time() - end)

        if args.gpu is not None:
            images[0] = images[0].cuda(args.gpu, non_blocking=True)
            images[1] = images[1].cuda(args.gpu, non_blocking=True)

        with autocast(device_type="cuda"):
            # compute output
            lossall, output, target = model(
                im_q=images[0], 
                im_k=images[1],
                criterion=criterion,
                epoch_ratio=epoch/args.epochs
                )
         
        loss=lossall[0]
        loss_full=lossall[1] if len(lossall)>=2 else torch.tensor(0.0)
        loss_next=lossall[2] if len(lossall)>=3 else torch.tensor(0.0)
        loss_clip=lossall[3] if len(lossall)>=4 else torch.tensor(0.0)
        # acc1/acc5 are (K+1)-way contrast classifier accuracy
        # measure accuracy and record loss
 
        acc1, acc5 = accuracy(output, target, topk=(1, 5))
        losses.update(loss.item(), images[0].size(0))
        losses_full.update(loss_full.item(), images[0].size(0))
        losses_next.update(loss_next.item(), images[0].size(0))
        losses_clip.update(loss_clip.item(), images[0].size(0))
        top1.update(acc1[0], images[0].size(0))
        top5.update(acc5[0], images[0].size(0))

        # compute gradient and do SGD step
        optimizer.zero_grad()


        scaler.scale(loss).backward()
        scaler.step(optimizer)
        scaler.update()

        # loss.backward()
        # clip_grad_norm_(model.parameters(), max_norm=1.0)
        # optimizer.step()

        # measure elapsed time
        batch_time.update(time.time() - end)
        end = time.time()
        dist.barrier()
        if i % args.print_freq == 0:
            progress.display(i)
 
    if epoch % args.knn_freq == 0:
        backup_model_fc = model.module.encoder_q.fc
        model.module.encoder_q.fc = nn.Identity()
        backup_model_sum_output = model.module.encoder_q.sum_output
        model.module.encoder_q.sum_output = False
        knn_acc1, knn_acc5, knn_std, knn_temporal_feature_similarity = knn_test(model.module.encoder_q, knn_train_loader, test_loader, args)
        model.module.encoder_q.sum_output = backup_model_sum_output
        model.module.encoder_q.fc = backup_model_fc
        if not args.multiprocessing_distributed or (
            args.multiprocessing_distributed and args.rank % args.ngpus_per_node == 0
        ):        
            wandb.log({
                "knn_acc1": knn_acc1,
                "knn_acc5": knn_acc5,
                "knn_std": knn_std,
                "knn_temporal_feature_similarity": knn_temporal_feature_similarity,
            },step=epoch)

    if losses.avg < args.best_losses:
        args.best_losses = losses.avg
        args.best = True
    else:
        args.best = False
    if not args.multiprocessing_distributed or (
            args.multiprocessing_distributed and args.rank % args.ngpus_per_node == 0
        ):        
        wandb.log({
            "epoch": epoch,
            "losses": losses.avg,
            "losses_full": losses_full.avg,
            "losses_next": losses_next.avg,
            "losses_clip": losses_clip.avg,
            "top1": top1.avg,
            "top5": top5.avg,
            "lr": optimizer.param_groups[0]['lr'],
        })



if __name__ == "__main__":
    main()
